/*
* \file: errmemd_backend_file.c
*
* Definition of the backend which handles outptut to a file.
* 
* Implementation of an error memory backend that stores error memory
* messages in text file(s). The file(s) are size-limited and have
* a continuous increasing index. At a maximum the specified number of
* of files, including potentially specified persistent files, exist in
* parallel.
* The file storage resides in <path> and has in principal the following parts:
*
* Configuration file   : file_storage_name.cfg
*
* With 
*  p = number of persistent files   p >= 0
*  n = number of maximum files      p < n    (mandatory)
*  c = currently highest file index c >= n-1 (maximum number of files in use)
*  if (c < n-1 ) ==> not all files in use ==> file_storage_name_<0..c> exist.
* 
* Persistent files     : file_storage_name_<0..p-1>  ;  p > 0
* Not persistent files : file_storage_name_<c-n+p+1> ... file_storage_name_<c>
*
* \component: errmemd
*
* \author: Kai Tomerius (ktomerius@de.adit-jv.com)
*          Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/

#include <ctype.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <syslog.h>
#include <errno.h>
#include <libgen.h>
#include <dirent.h>

#include "errmemd.h"
#include "errmem_backend.h"
#include "errmem_socket_interface.h"

const char FILE_SEPARATOR = '_';

typedef struct ErrmemBackendFileCfg {
	uint32_t size;              // maximum file size of each file
	uint32_t nr_files;          // number of maximum parallel files
	uint32_t persistent;        // number of non-overwritable files
} ErrmemBackendFileCfg_t;

// the error memory backend
typedef struct ErrmemBackendFile {
	ErrmemBackend_t        backend;     // error memory backend interface
	int32_t                fd;          // handle to file to write to
	int32_t                type;        // type of backend storage
	uint32_t               msg_magic;   // start of a message
	ErrmemBackendFileCfg_t cfg;         // configuration of file storage
	uint32_t               status;      // status during storage validation
	uint32_t               access_mode; // access rights to storage
	uint32_t               index_low;   // index of the oldest file in use
	uint32_t               index_high;  // index of the current file
	uint32_t               cnt_files;   // number of existing files in storage
	uint32_t               nr;          // position of device in cmd line
	int32_t                def;         // indicates device is default or not
	uint32_t               storage;     // monotonously increasing msg number
	uint32_t               flags;       // eliminate already interpreted flags
	uint16_t               seed;        // Base for CRC calculation
	off_t                  file_size;   // file size based on filesystem blksize
	char*                  name_idx;    // memory for file name + file index
	char*                  cur_file;    // complete name of current storage file
	char*                  name;        // file name
	char*                  path;        // file path
	char*                  command;     // cmd to execute on setup of backend
} ErrmemBackendFile_t;

/* This function extracts the index of the file from its filename idx_fn if
 * this file is part of the file storage referenced by fn.
 * If fn is only a part somewhere in the middle of the filename idx_fn or if
 * fn does not match at all or the numbered index has trailing or 
 * preceding characters or characters in between, this function 
 * return idx == ~0, otherwise it returns an unsigned integer >= 0.
 * Example: idx_fn = myStorage.txt.log_007
 *              fn = myStorage.txt.log
 * Return will be 7.
 * Not allowed examples : idx_fn = mmyStorage.txt.log_007
 *                    or: idx_fn = mStorage.txt.log
 *                    or: idx_fn = mStorage.txt.log_007.txt
 */
static char* get_command(ErrmemBackend_t* backend)
{
	if (backend)
		return ((ErrmemBackendFile_t*)backend)->command;
	return NULL;
}

static char* get_name(ErrmemBackend_t* backend)
{
	if (backend)
		return ((ErrmemBackendFile_t*)backend)->name;
	return NULL;
}

static unsigned get_nr(ErrmemBackend_t* backend)
{
	if(backend)
		return ((ErrmemBackendFile_t*)backend)->nr;
	return ~0;
}

static void set_nr(ErrmemBackend_t* backend, uint32_t pnr)
{
	if(backend)
		((ErrmemBackendFile_t*)backend)->nr = pnr;
	return;
}

static int32_t is_default(ErrmemBackend_t* backend)
{
	if(backend)
		return ((ErrmemBackendFile_t*)backend)->def;
	return 0;
}

static void set_default(ErrmemBackend_t* backend, int32_t val)
{
	if(backend)
		((ErrmemBackendFile_t*)backend)->def = val;
	return;
}

static unsigned get_type(ErrmemBackend_t* backend)
{
	if(backend)
		return ((ErrmemBackendFile_t*)backend)->type;
	return 0;
}

static int32_t get_idx_from_fn(char* idx_fn, char* fn, uint32_t* id)
{
	int32_t  err =  0;
	*id = ~0;
	if (idx_fn && fn) {
		char* st = strstr(idx_fn, fn);
		char* end = idx_fn + strlen(idx_fn);
		/* Filename matches at the beginning */
		if (st && st == idx_fn) {
			/* set st behind the name to check separator */
			st += strlen(fn);
			if ((st < end) && *st == FILE_SEPARATOR) {
				/* set st to the first expected digit */
				st++;
				*id = 0;
				/* now check the end */
				while (st < end) {
					if (*st >= '0' && *st <= '9')
						*id = *id * 10 + (*st - '0');
					else {
						*id = ~0;
						break;
					}
					st++;
				}
			}
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", "Parameter check failed in function "
			   "get_idx_from_fn. Error = ", strerror(-err));
	}
	return err;
}

/* This function concatenates a number of nr strings passed by strings and
 * returns the result in res.
 * The memory pointed to by *res at call time is freed by this function.
 * The memory pointed to by *res at return time has to be freed by the caller.
 * Null pointer in the strings array will be ignored.
*/
static int32_t concatenate(int32_t nr, char** strings, char** res)
{
	int32_t err = 0;
	int32_t i   = 0;
	int32_t len = 0;
	if (nr > 0 && strings) {
		for (; i < nr; i++) {
			if (strings[i])
				len += strlen(strings[i]);
		}
		len += 1;
		if (*res) {
			free(*res);
			*res = NULL;
		}			
		*res = (char*)calloc(1, sizeof(char)*len);
		if (*res) {
			for (i = 0; i < nr; i++) {
				if (strings[i])
					*res = strcat(*res, strings[i]);
			}
		} else {
			err = -ENOMEM;
			syslog(LOG_CRIT, "%s %s",
				   "Failed to concatenate strings - error = ",
				   strerror(-err));
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s",
			   "Parameter check failed in function  concatenate - error = ",
			   strerror(-err));
	}
	return err;
}

/* This function returns the current filename, means the filename with the
 * currently highest index in dest. The filename is created using path, name and
 * value of index_high from dev structure */
static int32_t get_current_fn(ErrmemBackendFile_t* dev, char** dest)
{
	int32_t err = 0;
	if (dev && dev->path && dev->name_idx && dev->name && ~dev->index_high) {
		char* s[] = {dev->path, "/", dev->name_idx};
		sprintf(dev->name_idx + strlen(dev->name), "%c%u",
				FILE_SEPARATOR, dev->index_high);
		err = concatenate(3, s, dest);
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s",
			   "Parameter check failed in function get_current_fn - error =",
			   strerror(-err));
	}
	return err;
}

/* This function returns the filename with index id in *dest*/
static int32_t get_storage_fn(ErrmemBackendFile_t* dev, uint32_t id, char** dest)
{
	int32_t err = 0;
	char* s[] = {dev->path, "/", dev->name_idx};
	sprintf(dev->name_idx + strlen(dev->name), "%s%u",
				(char*)&FILE_SEPARATOR, id);
	err = concatenate(3, s, dest);
	return err;
}

/* If cur is ~0 the filename with the lowest existing index is returned 
 * If cur is not ~0 the next filename with a valid index > cur is returned 
 */
static int32_t get_next_fn(ErrmemBackendFile_t* dev, char** dest, uint32_t cur)
{
	int32_t err = 0;
	if (dev && dev->path && dev->name_idx && dev->name) {
		char* s[] = {dev->path, "/", dev->name_idx};
		/* don't check top index because a new file could be requested */
		/* get the very first file of the file storage */
		if (!~cur) {
			if (dev->cfg.persistent)
				cur = 0;
			else if (!~dev->index_low)
				cur = 0;
			else
				cur = dev->index_low;
		} else {
			/* get the next possible index */
			if (dev->cfg.persistent && cur < dev->cfg.persistent - 1)
				cur++;
			else if (!~dev->index_low)
				cur++;
			else if (~dev->index_low && cur + 1 <= dev->index_low)
				cur = dev->index_low;
			else
				cur++;
		}

		sprintf(dev->name_idx + strlen(dev->name), "%s%u",
				(char*)&FILE_SEPARATOR, cur);
		err = concatenate(3, s, dest);
		if (!err)
			err = cur;
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s",
			   "Parameter check failed in function get_next_fn - error =",
			   strerror(-err));
	}
	return err;
}

static int32_t remove_file(char* path, char* name, char* ext) 
{
	int32_t err  = 0;
	char*   s[]  = {path, "/", name, ext};
	char*   fcfg = NULL;
	struct stat st;

	if (!path)
		s[1] = NULL;

	err = concatenate(4, s, &fcfg);
	if (!err && fcfg) {
		err = stat(fcfg, &st);
		if (!err) {
			err = remove(fcfg);
			if (-1 == err) {
				err = -errno;
				syslog(LOG_CRIT, "%s %s %s %s",
					   "Cannot remove file", fcfg, "- error =",
					   strerror(-err));
			}
		} else {
			err = -errno;
			if (errno != ENOENT)
				syslog(LOG_CRIT, "%s %s",
					   "Cannot stat existing config file - error =",
					   strerror(-err));
		}
	}
	if (fcfg)
		free(fcfg);

	return err;
}

static int32_t create_config(ErrmemBackendFile_t* dev)
{
	int32_t err = 0;
	char* s[]  = {dev->path, "/", dev->name, ".cfg"};
	char* fcfg = NULL;
	struct stat st;

	if(!dev->path)
		s[1] = NULL;

	err = concatenate(4, s, &fcfg);
	if (!err && fcfg) {
		err = stat(fcfg, &st);
		if (!err) {
			err = remove(fcfg);
			if (-1 == err) {
				err = -errno;
				syslog(LOG_CRIT, "%s %s",
					   "Cannot remove exiting config file - error =",
					   strerror(-err));
				dev->status = STAT_ABORT;
			}
		} else if (errno != ENOENT) {
			err = -errno;
			syslog(LOG_CRIT, "%s %s",
				   "Cannot stat existing config file - error =",
				   strerror(-err));
			dev->status = STAT_ABORT;			
		} else
			err = 0;

		if (!err) {
			err = open_dev(fcfg, O_WRONLY| O_CREAT, 0);
			if (-1 == err) {
				err = -errno;
				syslog(LOG_CRIT, "%s %s %s %s", "Cannot create config file",
					   fcfg, "error =", strerror(-err));
				dev->status = STAT_ABORT;
			} else {
				int32_t fd = err;
				err = write_dev(fd, fcfg, (char*)&dev->cfg, sizeof(ErrmemBackendFileCfg_t));
				if(err < 0)
					dev->status = STAT_ABORT;

				err = close_dev(fd, fcfg);
				if (err < 0)
					dev->status = STAT_ABORT;
			}
		}
	}
	if (fcfg)
		free(fcfg);

	if (err >= 0) {
		dev->status = STAT_OK;
		err = 0;
	}
	return err;
}

// info - show information about this backend
static void info(struct ErrmemBackend* backend, int fd)
{
	ErrmemBackendFile_t* dev = (ErrmemBackendFile_t*) backend;

	if (dev && dev->name && dev->path) {
		dprintf(fd, "Path: %s - File storage: %s: - Number of Files: %u - Persistent: %u - Maximum size: %u bytes\n",
				dev->path, dev->name, dev->cfg.nr_files, dev->cfg.persistent, dev->cfg.size);
		dprintf(fd, "Current file: %s\n", dev->cur_file);
	}
	return;
}

/* finds storage files and deletes them */
static int32_t erase(struct ErrmemBackend* backend)
{
	int32_t err         = 0;
	DIR*    dirp        = NULL;
	struct dirent* dent = NULL;

	ErrmemBackendFile_t* dev = (ErrmemBackendFile_t*) backend;

	if (dev && dev->path && dev->name) {
		err = remove_file(dev->path, dev->name, ".cfg");
		if (!err || err == -ENOENT) {
			err = 0;
			dirp = opendir(dev->path);
			if (!dirp) {
				dev->status = STAT_ABORT;
				err = -errno;
				syslog(LOG_CRIT, "%s %s %s %s",
					   "Failed to open directory", dev->path, "- error =",
					   strerror(-err));
			}
			if (!err) {
				do {
					errno = 0;
					dent = readdir(dirp);
					if (dent) {
						uint32_t id;
						err = get_idx_from_fn(dent->d_name, dev->name, &id);
						if (err < 0)
							break;
						if (~id) {
							char* dfp[] = {dev->path, "/", dent->d_name};
							char* df    = NULL;
							err = concatenate(3, dfp, &df);
							if (!err && df) {
								err = remove(df);
								if (err) {
									err = -errno;
									syslog(LOG_CRIT, "%s %s %s %s",
										   "Failed to remove existing storage "
										   "file", df, "- error =",
										   strerror(-err));
								}
							} else if (!err) {
								err = -EIO;
								syslog(LOG_CRIT, "%s %s",
									   "Failed to concatenate file name - "
									   "Internal application error: ", 
									   strerror(-err));
							}
							if (df)
								free(df);
						} /* file is not part of storage - nothing to do */
					} else if (errno != 0) {
						err = -errno;
						syslog(LOG_CRIT, "%s %s %s %s",
							   "Failed to read directory", dev->path, "error =",
							   strerror(-err));
					}
					if (err < 0) {
						dev->status = STAT_ABORT;
						break;
					}
				} while (dent != NULL);

				err = closedir(dirp);
				if (err) {
					dev->status = STAT_ABORT;
					err = -errno;
					syslog(LOG_CRIT, "%s %s %s %s",
						   "Failed to close directroy", dev->path,
						   "- error =", strerror(-err));			
				}
				if (!err) {
					dev->status = STAT_ERASED;
					dev->index_high = 0;
					dev->index_low  = ~0;
					dev->cnt_files  = 0;
					dev->file_size  = ~0;
				}
			}
		} else 
			dev->status = STAT_ABORT;
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", 
			   "Parameter check failed in function erase - error =",
			   strerror(-err));		
	}
	return err;
}

static int32_t split_name(const char* long_path, char** path, char** fn)
{
	int32_t err = 0;
	char*   cp  = NULL;
	char*   cp2 = NULL;
	char*   p   = NULL;
	char*   f   = NULL;
	int     len = 0;
	if (long_path) {
		len = strlen(long_path);
		if (long_path[len-1] == '/'  ||
			long_path[len-1] == '\\' ||
			long_path[len-1] == '.'  ||
			long_path[len-1] == ' '  ||
			long_path[len-1] == '\0' ||
			long_path[len-1] == '\n') {
			err = -EINVAL;
			syslog(LOG_CRIT, "%s %d %s %s",
				   "Storage name has invalid character", long_path[len-1],
				   "at the end - error = ", strerror(-err));
		}
		if(!err) {
			/* Do not modify long_path. It has to be freed outside */
			cp  = strndup(long_path, strlen(long_path));
			cp2 = strndup(long_path, strlen(long_path));
			if (!cp || !cp2) {
				err = -errno;
				syslog(LOG_CRIT, "%s %s",
					   "Cannot duplicate file name - error = ",
					   strerror(-err));
			} else {
				f = basename(cp);
				p = dirname(cp2);
				if (p) {
					if (!strlen(p) || p[0] != '/') {
						err = -EINVAL;
						syslog(LOG_CRIT, "%s %s",
							   "No path info given - error = ",
							   strerror(-err));
					}
				} else {
					err = -EINVAL;
					syslog(LOG_CRIT, "%s %s",
						   "No path info given - error = ",
						   strerror(-err));
				}
			}
			if (!err && path && p) {
				*path = strndup(p, strlen(p));
				if (!*path) {
					err = -errno;
					syslog(LOG_CRIT, "%s %s",
						   "Cannot duplicate path name - error = ",
						   strerror(-err));				
				}
			}
			if (!err && fn && f) {
				*fn = strndup(f, strlen(f));
				if (!*fn) {
					err = -errno;
					syslog(LOG_CRIT, "%s %s",
						   "Cannot duplicate pure file name - error = ",
						   strerror(-err));				
				}
			}
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", 
			   "Parameter check failed in function split_name - error =",
			   strerror(-err));
	}
	if (cp)
		free(cp);
	if (cp2)
		free(cp2);
	return err;
}

/* This function checks the current storage regarding the amount of 
 * existing and number of specified files, considering potentially
 * specified persistent files and determines and checks the lowest
 * allowed index of a file and deletes an existing one with a lower
 * index in case of necessity. The file with the highest index which
 * name is stored in dev->cur_file will be created and the
 * corresponding file handle is stored in dev->fd.
 * This function takes care that in case the amount of existing files
 * would exceed the specified amount of files, the oldest not
 * persistent file is deleted first before the new one is created.
 */
static int32_t re_organize_storage(ErrmemBackendFile_t* dev)
{
	int32_t  err  =  0;
	uint32_t id   = ~0;
	uint32_t del  = ~0;
	char*    nidx = NULL;
	if (dev && dev->name && dev->path && dev->name_idx && dev->cur_file) {
		/* Get the file name of the potentially new file from dev->cur_file.
		 * The file name will be stored in dev->name_idx.*/
		err = split_name(dev->cur_file, NULL, &nidx);
		if (!err) {
			/* get the index from the filename */
			err = get_idx_from_fn(nidx, dev->name, &id);
			if (!err) {
				if (id != dev->index_high) {
					err = -EBADF;
					syslog(LOG_CRIT, "%s %s",
						   "File index does not match current high index" 
						   " - error=", strerror(-err));
				}					
				if (!err && ~id) {
					/* check the storage and check low index */
					if (dev->cfg.persistent && id < dev->cfg.persistent) {
						if (~dev->index_low) {
							err = -EBADF;
							syslog(LOG_CRIT, "%s %s",
								   "Inconsistent storage. Index low exists " 
								   "- error=", strerror(-err));
						}
						/* leaving the persistent file area 
						 * if index low is set that would be inconsistent */
					} else if (dev->cfg.persistent && id == dev->cfg.persistent) {
						if (~dev->index_low) {
							err = -EBADF;
							syslog(LOG_CRIT, "%s %s",
								   "Inconsistent storage. Index low exists " 
								   "- error=", strerror(-err));
						} else
							/* This is the first file which is not persistent.
							 * Low index has to be set here the first time.
							 * Otherwise the storage already exists and files
							 * other than persistent has already been created. */
							dev->index_low = dev->cfg.persistent;
					} else if (!id) {
						if (~dev->index_low) {
							err = -EBADF;
							syslog(LOG_CRIT, "%s %s",
								   "Inconsistent storage. Index low exists " 
								   "- error=", strerror(-err));
						} else
							/* This is the first file which is not persistent.
							 * Low index has to be set here the first time.
							 * Otherwise the storage already exists and files
							 * other than persistent has already been created. */
							dev->index_low = 0;
					} else if (id < dev->cfg.nr_files) {
						/* In case id equals 1, index_low is not allowed to have
						 * a value greater zero and other than 0xFFFFFFFF 
						 * Check here this special case because the first call to
						 * this function may have id = 1 and no persistent parts */
						if (id == 1 ) {
							if (dev->index_low && ~dev->index_low) { 
								err = -EBADF;
								syslog(LOG_CRIT, "%s %s",
									   "Inconsistent storage. Index low exists " 
									   "- error=", strerror(-err));
							} else
								dev->index_low = 0;								
						}
					} else {
						if(id >= dev->cfg.nr_files) {
							/* del is the lowest index which has to be deleted as function of 
							 * dev->index_high.*/
							del = (dev->index_high - ((dev->cfg.nr_files - dev->cfg.persistent) - 1)) - 1;
							/* should normally not happen that index of currently 
							 * oldest existing non-persistent file is greater
							 * than the lowest index which is not allowed to exist
							 * any longer */
							if (dev->index_low > del) {
								err = -EBADF;
								syslog(LOG_CRIT, "%s %s",
									   "Inconsistent storage. Index low greater " 
									   "than lower border- error=", strerror(-err));
							}
							else if (dev->index_low == del)
								/* the latest file has to be deleted. Index stored in del */
								dev->index_low++;
							else {
								err = -EBADF;
								syslog(LOG_CRIT, "%s %s",
									   "Inconsistent storage. Index low is less " 
									   "than lower border- error=", strerror(-err));
							}
						}
					}
					if (!err) {
						if (~del) {
							char* name = NULL;
							err = get_storage_fn(dev, del, &name);
							if (!err) {
								err = remove_file(NULL, name, NULL);
								if (!err)
									dev->cnt_files--;
							}
							if (name)
								free(name);
						}
						/* create file index high*/
						if (!err) {
							err = get_storage_fn(dev, dev->index_high, &dev->cur_file);
							if (!err) {
								err = open_dev(dev->cur_file, dev->access_mode | O_CREAT, 0);
								if (err >= 0) {
									struct stat s = {0};
									dev->fd = err;
									dev->cnt_files++;
									err = stat(dev->cur_file, &s);
									if (-1 == err) {
										err = -errno;
										syslog(LOG_CRIT, "%s %s %s %s",
											   "Stat failed on device",dev->cur_file,
											   "- error =", strerror(-err));
									} else if (!s.st_blksize) {
										err = -EIO;
										syslog(LOG_CRIT, "%s %s %s %s",
											   "Can't determine blksize used by filesystem "
											   "residing", dev->path,
											   "- error =", strerror(-err));
									} else {
										dev->file_size = dev->cfg.size -
											((off_t)dev->cfg.size % s.st_blksize);
										dev->seed = calc_crc16(0, (void*)dev->cur_file,
															   strlen(dev->cur_file));
									}
								}
							}
						}
					}
				}
			} else {
				err = -ENOSTR;
				syslog(LOG_CRIT, "%s %s",
					   "Current file is not a regular part of storage - error=",
					   strerror(-err));
			}
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s",
			   "Parameter check failed in function re_organize_storage - error =",
			   strerror(-err));
	}

	if (nidx)
		free(nidx);

	return err;
}

static int32_t update_storage(ErrmemBackendFile_t* dev)
{
	int32_t err = 0;
	if (dev && dev->cur_file && dev->name && dev->path) {
		err = close_dev(dev->fd, dev->cur_file);
		if (!err) {
			dev->fd = -1;
            /* Function get_next_fn cares for freeing pointer cur_file in case
			 * no error occured before assigning the new char pointer.
			 * dev->cur_file has new index. This should be index_high + 1 */
			err = get_next_fn(dev, &dev->cur_file, dev->index_high);
			if (err < 0) {
				err = -EIO;
				syslog(LOG_CRIT, "%s %s%s%s %s %s",
					   "Can't create new file in file storage",
					   dev->path, "/", dev->name, "to store error messages - error:",
					   strerror(-err));
			} else {
				if (dev->cur_file) {
					dev->index_high = err;
					err = re_organize_storage(dev);
				} else {
					err = -EIO;
					syslog(LOG_CRIT, "%s %s%s%s %s %s",
						   "Internal application error creating new file in",
						   dev->path, "/", dev->name, "- error:",
						   strerror(-err));
				}
			}
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", 
			   "Parameter check failed in function update_storage - error =",
			   strerror(-err));
	}
	return err;
} 

// store - log an error memory message
static unsigned store(struct ErrmemBackend* backend, int len,
		  struct errmem_message* msg)
{
	ErrmemBackendFile_t* dev = (ErrmemBackendFile_t*) backend;
	PersistentMessage_t pmsg = {.magic = dev->msg_magic, .crc = 0,
								.reserved = ~0, .blocknum = 0,
								.len = len, .msg = msg};
	uint32_t seqnum          = 0;
	uint32_t flags           = msg->internal.flags;
	int32_t err              = 0;
	off_t pos                = 0;
	char buf[offsetof(PersistentMessage_t, msg) +
			 offsetof(struct errmem_message, message) + ERRMEM_MAX_ENTRY_LENGTH];

	if (dev) {
		if (dev->status == STAT_ERASED) {
			err = create_config(dev);
			if (err < 0)
				seqnum = ~0;
			else {
				err = get_current_fn(dev, &dev->cur_file);
				if (err < 0) 
					seqnum = ~0;
				else {
					err = open_dev(dev->cur_file, dev->access_mode | O_CREAT, 0);
					if (err < 0)
						seqnum = ~0;
					else {
						struct stat s = {0};
						dev->fd = err;
						dev->cnt_files = 1;
						err = stat(dev->cur_file, &s);
						if (-1 == err) {
							err = -errno;
							syslog(LOG_CRIT, "%s %s %s %s",
								   "Stat failed on device",dev->cur_file,
								   "- error =", strerror(-err));
						} else if (!s.st_blksize) {
							err = -EIO;
							syslog(LOG_CRIT, "%s %s %s %s",
								   "Can't determine blksize used by filesystem "
								   "residing", dev->path,
								   "- error =", strerror(-err));
						} else {
							dev->file_size =
								dev->cfg.size -
								((off_t)dev->cfg.size % s.st_blksize);
							dev->status = STAT_OK;
						}
					}
				}
			}
		}
		if (err < 0)
			dev->status = STAT_ABORT;

		if (dev->status == STAT_OK) {
			if (dev->cur_file) {
				err = seek_dev(dev->fd, dev->cur_file, 0, SEEK_CUR);
				if (err < 0) {
					seqnum = ~0;
					dev->status = STAT_ABORT;
				} else {
					if (!dev->seed)
						dev->seed = calc_crc16(0, (void*)dev->cur_file,
											   strlen(dev->cur_file));
					pos = (off_t)err;
					err = 0;
				}
			} else {
				err = -ENODEV;
				syslog(LOG_CRIT, "%s %s", "No current file to write to in store"
					   "function - error =", strerror(-err));
				dev->status = STAT_ABORT;
				seqnum = ~0;
			}
		}
		if (!err) {
			/* check whether message fits into the current file */
			if ((off_t)(pos + offsetof(PersistentMessage_t, msg) + pmsg.len) >
				dev->file_size) {
				err = update_storage(dev);
				if (err < 0) {
					dev->status = STAT_ABORT;
					seqnum = ~0;
				}
			}
		}
		
		if (!err) {
			pmsg.blocknum = dev->index_high;
			pmsg.crc = calc_crc16(dev->seed, (void*)&pmsg, 
								  offsetof(PersistentMessage_t, msg));
			
			memcpy((char*)&buf[0], (char*)&pmsg, offsetof(PersistentMessage_t, msg));

			if (err < 0) {
				dev->status = STAT_ABORT;
				seqnum = ~0;
			} else {
				// delete some flags in successive messages
				msg->internal.flags &=
					~(dev->flags & (ERRMEM_FLAG_RESTART |
									ERRMEM_FLAG_INITIALIZED |
									ERRMEM_FLAG_DROPPED));

				msg->internal.storage = ++dev->storage;
				msg->internal.crc = 0;
				msg->internal.crc = calc_crc16(dev->seed, 
											   (void*)pmsg.msg, pmsg.len);

				memcpy((char*)&buf[offsetof(PersistentMessage_t, msg)],
								   (char*)pmsg.msg, pmsg.len);

				err = write_dev(dev->fd, dev->cur_file, (char*)&buf[0], 
								offsetof(PersistentMessage_t, msg) + pmsg.len);
				if (err < 0) {
					dev->status = STAT_ABORT;
					seqnum = ~0;
				} else {
					seqnum = msg->internal.seqnum;
					dev->flags = flags;
				}
			}
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", 
			   "Parameter check failed in function store - error =",
			   strerror(-err));		
		seqnum = ~0;
	}

	return seqnum;
}

// destroy - free resources
static void destroy(struct ErrmemBackend* backend)
{
	ErrmemBackendFile_t* dev = (ErrmemBackendFile_t*)backend;

	if (dev) {
		if (dev->fd)
			close(dev->fd);

		if (dev->name)
			free(dev->name);

		if (dev->path)
			free(dev->path);
		
		if (dev->name_idx)
			free(dev->name_idx);

		if (dev->cur_file)
			free(dev->cur_file);

		if (dev->command)
			free(dev->command);

		free(dev);
	}
}

/* This function checks the given command line parameters with regard
 * to completeness and plausibility
*/
static int32_t check_parms(ErrmemBackendFile_t* dev)
{
	int32_t err = 0;
	if (dev) {
		if (dev->access_mode != O_RDONLY) {
			if (!dev->cfg.size) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s",
					   "File device: Size information missing "
					   "- check -b --bs option  - error =", strerror(-err));
			}
			if (!dev->cfg.nr_files) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s",
					   "File device: Specification of number of "
					   "files invalid (0 files)- check -c --cnt option  - error =",
					   strerror(-err));			
			}
			if (!~dev->cfg.persistent) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s",
					   "File device: Specification of number of persistent "
					   "files missing - check -e --persistent option "
					   " - error =", strerror(-err));			
			}
			if (!err && dev->cfg.persistent >= dev->cfg.nr_files) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s",
					   "File device: Number of persistent files exceeds "
					   "number of total files. Check options -e --persistent "
					   "and -c --cnt - error =", strerror(-err));
			}
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", "File device: Parameter check failed in "
			   "function check_parms  - error =", strerror(-err));
	}
	return err;
}

/* Check existence of file device config file */
static int32_t check_ex_file_dev_cfg(char* name, char* path)
{
	int32_t err = 0;
	char* s[] = {path, "/", name, ".cfg"};
	char* fcfg = NULL;
	struct stat st = {0};
	err = concatenate(4, s, &fcfg);
	if (!err && fcfg)
		err = stat(fcfg, &st);
	if (fcfg)
		free(fcfg);
	return err;
}

/* This function gets the configuration of an existing file storage 
 * In case of success it returns 0.
 * If -ENOENT is returned the configuration could not be found.
 * Other error are returned as negative values 
 */
static int32_t get_file_device_cfg(char* name, char* path,
								   ErrmemBackendFileCfg_t* cfg)
{
	int32_t err = 0;
	char* s[] = {path, "/", name, ".cfg"};
	char* fcfg = NULL;
	int fh;
	err = concatenate(4, s, &fcfg);
	if (!err && fcfg) {
		err = open_dev(fcfg, O_RDONLY, 1);
		if (err >= 0) {
            fh = err;
			err = read_dev(fh, fcfg, (char*)cfg, sizeof(ErrmemBackendFileCfg_t));
			close(fh);
		} else
			err = -errno;
	}
	if (fcfg)
		free(fcfg);
	return err;
}

/* This function figures out whether the file storage is consistent or not.
 * It gets the amount of already existing files and checks that against the
 * configuration. This includes the potentially specified persistent files.
 * It looks for gaps in the file sequence and checks the size of completed
 * files to identify possible corruption.
 * In case of any hint for corruption, either indicated by the missing of
 * expected files or not plausible size information the storage will be
 * ignored.
 *
 * The check of creation and modification time should be handled with care.
 * As long as on the target no UTC timestamp is available, only a timestamp
 * based on the target startup time is available. This may lead to not correct
 * creation or modification times and to misinterpretation of them.
 * ==> Time check will not be implemented here.
 */
static int32_t check_file_storage(ErrmemBackendFile_t* dev)
{
	int32_t  err = 0;
	uint32_t cur = 0;

	if (dev && dev->name_idx) {
		/* check the indices got from directory check */
		if (dev->cnt_files > dev->cfg.nr_files) {
			err = -ENFILE;
			syslog(LOG_CRIT, "%s %d %s %d %s %s", "File storage consists of",
				   dev->cnt_files, "but only", dev->cfg.nr_files,
				   "files configured - Error =", strerror(-err));
			dev->status = STAT_ABORT;
		}
		if (!err) {
			if (!~dev->index_low && (dev->index_high > 
									 (dev->cfg.nr_files - 1))) {
				err = -ENFILE;
				syslog(LOG_CRIT, "%s %d %s %d %s %s",
					   "Inconsistent file storage - Highest Index is",
					   dev->index_high, "but only", dev->cfg.nr_files,
					   "files configured and no gaps detected - Error =",
					   strerror(-err));
				dev->status = STAT_ABORT;
			}
		}
		if (!err) {
			if (~dev->index_low &&
				((int32_t)dev->index_low < (int32_t)(dev->index_high - 
									(dev->cfg.nr_files +
										dev->cfg.persistent + 1)))) {
				err = -ENFILE;
				syslog(LOG_CRIT, "%s %s %s %s %d %s %d %s %d %s %s",
					   "Inconsistent file storage in", dev->path, "\n",
					   "- Gap between highest and lowest index too big - high:",
					   dev->index_high, "low:", dev->index_low,
					   "files configured:", dev->cfg.nr_files,
					   " - Error =", strerror(-err));
				dev->status = STAT_ABORT;
			}
		}

		if (!err) {
			/* set the correct begin of expected indices in the file storage */
			if (dev->cfg.persistent)
				cur = 0;
			else {
				if (~dev->index_low)
					cur = dev->index_low;
				else if (dev->index_high >= dev->cfg.nr_files - 1)
					cur = dev->index_high - (dev->cfg.nr_files + dev->cfg.persistent + 1);
				else
					cur = 0;
			}

			while ((cur < dev->cfg.persistent) || (cur <= dev->index_high)) {
				struct stat s;
				char* name  = NULL;
				/* check here for highest index. File must not already be created */
				if (cur > dev->index_high)
					break;
				err = get_storage_fn(dev, cur, &name);
				if (err || !name)
					break;
				err = stat(name, &s);
				if (!err) {
					/* store the file size based on the configured file size  
					 * and the blocksize of the underlying file system */
					if (!~dev->file_size && s.st_blksize)
						dev->file_size = dev->cfg.size -
							((off_t)dev->cfg.size % s.st_blksize);

					if (!~dev->file_size) {
						err = -EIO;
						syslog(LOG_CRIT, "%s %s %s %s", "Can't determine blocksize of"
							   "of filesystem residing", dev->path, "error ",
							   strerror(-err));
						dev->status = STAT_ABORT;
					}
					/* cur index exists */
					if (!err && dev->index_high > cur) {
						/* check the filesize only in case a file with higher
						   index exists */
						if (s.st_size > dev->file_size) {
							err = -EFBIG;
							syslog(LOG_INFO, "%s %s %s %s",
								   "File", name,
								   "too big compared to size specification. "
								   "Erase storage before reuse. Error =",
								   strerror(-err));
							dev->status = STAT_ABORT;
						}
						if (s.st_size <
							(off_t)(dev->file_size - 2 * sizeof(struct errmem_message))) {
							err = -EIO;
							syslog(LOG_INFO, "%s %s %s %s",
								   "File", name,
								   "too small compared to size specification. "
								   "Erase storage before reuse. Error =",
								   strerror(-err));
							dev->status = STAT_ABORT;
						}
					}
					if (!err && cur >= dev->cfg.persistent && !~dev->index_low)
						dev->index_low = cur;
				} else if (errno == ENOENT) {
					if (cur < dev->cfg.persistent) {
						err = -errno;
						syslog(LOG_CRIT, "%s %s %s %s",
							   "Expected persistent file", name, 
							   "does not exist anymore. Erase storage before reuse."
							   " error = ", strerror(-err));
					} else {
						err = -errno;
						syslog(LOG_CRIT, "%s %s %s %s",
							   "Expected file", name, 
							   "does not exist anymore. Erase storage before reuse."
							   " error = ", strerror(-err));
					}
					dev->status = STAT_ABORT;
				} else {
					err = -errno;
					syslog(LOG_CRIT, "%s %s %s %s",
						   "Cannot stat expected file", dev->name_idx, 
						   "- error = ", strerror(-err));
					dev->status = STAT_ABORT;
				}

				free(name);

				if (err < 0)
					break;

				cur += 1;

				if (dev->cfg.persistent) {
					if (cur == dev->cfg.persistent) {
						if (~dev->index_low)
							cur = dev->index_low;
						else if (dev->index_high >= dev->cfg.nr_files - 1)
							cur = dev->index_high - (dev->cfg.nr_files +
								dev->cfg.persistent + 1);
					}
				}			
			}
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", 
			   "Parameter check failed in function check_and_build_file_storage"
			   " - error =", strerror(-err));
	} 
	return err;
}

static int32_t setup_dev (ErrmemBackendFile_t* dev)
{
	int32_t err         = 0;
	DIR*    dirp        = NULL;
	struct dirent* dent = NULL;
	if (dev && dev->path && dev->name) {
		/* Read the directory dev->path.
		 * If the file storage has to be created, delete all files which
		 * belongs to that file storage if any exists.
		 * If the file storage has to be erased, delete all files which
		 * belongs to that file storage if any exists.
		 * If the file storage seems to be ok, check the existing files
		 * with regard to the configuration. Get highest and lowest index
		 * and delete oldest files if not compliant with configuration. 
		 */
		if (dev->access_mode != O_RDONLY) {
			if (dev->command && (!strncmp((const char*)dev->command,
						      (const char*)"erase", strlen("erase"))))
				dev->status = STAT_EX_ERASE;
		}

		if (dev->status == STAT_CREATE ||
			dev->status == STAT_EX_ERASE) {
			err = erase(&dev->backend);
			if (err < 0) {
				dev->status = STAT_ABORT;
			}
		} else if (dev->status == STAT_OK) {
			dirp = opendir(dev->path);
			if (!dirp) {
				dev->status = STAT_ABORT;
				err = -errno;
				syslog(LOG_CRIT, "%s %s %s %s",
					   "Failed to open directory", dev->path, "- error =",
					   strerror(-err));
			}
			if (!err) {
				do {
					errno = 0;
					dent = readdir(dirp);
					if (dent) {
						uint32_t id;
						err = get_idx_from_fn(dent->d_name, dev->name, &id);
						if (err < 0)
							break;
						if (~id) {
							dev->cnt_files++;
							if (!~dev->index_high)
								dev->index_high = id;
							else if (id > dev->index_high)
								dev->index_high = id;

							if (!~dev->index_low) {
								if (id >= dev->cfg.persistent)
									dev->index_low = id;
							} else if (id < dev->index_low &&
									   id >= dev->cfg.persistent)
								dev->index_low = id;
						} /* file is not part of storage */
					} else if (errno != 0) {
						dev->status = STAT_ABORT;
						err = -errno;
						syslog(LOG_CRIT, "%s %s %s %s",
							   "Failed to read entry from directroy", dev->path,
							   "- error =", strerror(-err));
					}
					if (err < 0)
						break;
				} while (dent != NULL);
			}
		} else { 
			dev->status = STAT_ABORT;						
			err = -EINVAL;
			syslog(LOG_CRIT, "%s %s",
				   "Invalid status of existing file storage "
				   "- error =", strerror(-err));
		}

		if (dirp) {
			err = closedir(dirp);
			if (err) {
				dev->status = STAT_ABORT;
				err = -errno;
				syslog(LOG_CRIT, "%s %s %s %s",
					   "Failed to close directroy", dev->path,
					   "- error =", strerror(-err));			
			}
		}
			
		if ((dev->status == STAT_OK) && (!~dev->index_high)) {
			if (dev->access_mode != O_RDONLY) {
				err = remove_file(dev->path, dev->name, ".cfg");
				if (err < 0 && err != -ENOENT)
					dev->status = STAT_ABORT;
				else {
					dev->status = STAT_ERASED;
					dev->index_high = 0;
					dev->index_low  = 0;
				}
			} else {
				dev->status = STAT_ABORT;
				err = -ENODATA;
				syslog(LOG_CRIT, "%s %s %s %s",
					   "File storage in directroy", dev->path,
					   "is empty - error =", strerror(-err));
			}
		} else if (dev->status == STAT_OK)
			err = check_file_storage(dev);
		
		if (err < 0)
			dev->status = STAT_ABORT;

		if (dev->status == STAT_OK) {
			if (dev->access_mode != O_RDONLY) {
				err = get_current_fn(dev, &dev->cur_file);
				if (!err && dev->cur_file) {
					err = open_dev(dev->cur_file, O_RDWR, 0);
					if (err >= 0) {
						dev->fd = err;
						err = seek_dev(dev->fd, dev->cur_file, 0, SEEK_END);
						if (err < 0)
							dev->status = STAT_ABORT;
					} else
						dev->status = STAT_ABORT;
				} else
					dev->status = STAT_ABORT;
			} else {
				err = get_next_fn(dev, &dev->cur_file, ~0);
				if (!err && dev->cur_file) {
					err = open_dev(dev->cur_file, O_RDONLY, 0);
					if (err >= 0) {
						dev->fd = err;
						err = seek_dev(dev->fd, dev->cur_file, 0, SEEK_SET);
						if (err < 0)
							dev->status = STAT_ABORT;
					} else
						dev->status = STAT_ABORT;
				} else
					dev->status = STAT_ABORT;
			}
			if (dev->status == STAT_OK)
				dev->seed = calc_crc16(0, (void*)dev->cur_file,
									   strlen(dev->cur_file));
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", 
			   "Parameter check failed in function setup_dev - error =",
			   strerror(-err));		
	}
	return err;
}
/* create - creates a logfile error memory backend */
static int32_t create(int32_t acc, struct ErrmemBackend* next)
{
	int32_t err = 0;
	ErrmemBackendFileCfg_t cfg = {0};
	ErrmemBackendFile_t* dev   = (ErrmemBackendFile_t*)next;

	if (dev && dev->name && dev->path) {
		dev->access_mode = acc;
		
		/* check command line parameter */
		err = check_parms(dev);
		if (err >= 0)
			/* get file device configuration */
			err = get_file_device_cfg(dev->name, dev->path, &cfg);
		if (err >= 0) {
			/* In case the device shall only be read the
			 * configuration is take as it is.
			 */
			if (dev->access_mode == O_RDONLY) {
				dev->cfg = cfg;
				dev->status = STAT_OK;
			}
			else {
				if (dev->cfg.size != cfg.size ||
					dev->cfg.nr_files != cfg.nr_files ||
					dev->cfg.persistent != cfg.persistent)
					/* ignore the existing configuration */
					dev->status = STAT_CREATE;
				else
					/* configuration is already stored */
					dev->status = STAT_OK;
			}				
		} else if (err == -ENOENT) {
			if (dev->access_mode == O_RDONLY) {
				syslog(LOG_CRIT, "%s %s", "Config file not found - Client aborts"
					   " error =", strerror(-err));
				dev->status = STAT_ABORT;
			}
			else {
				dev->status = STAT_CREATE;
				err = 0;
			}
		} else
			dev->status = STAT_ABORT;

		/* Setup the file storage if status is other than STAT_ABORT */
		if (err >= 0)
			err = setup_dev(dev);

		if (err < 0)
			dev->status = STAT_ABORT;

	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", "File device: Parameter check failed in "
			   "function create  - error =", strerror(EINVAL));
	}
	return err;
}

static void read_message(SessionInfo_t* si)
{
	int32_t  err             = 0;
	PersistentMessage_t pmsg = {0};

	if (si && si->fn && si->backend) {
		
		if (!si->pos) {
			if (!~si->seq_last)
				si->seq_last = 0;
			else if (!~si->seq_new)
				si->seq_last = si->seq_next;

			si->seq_next = si->seq_num;
			si->seq_new = si->seq_num;
		}

		/* read header */
		err = read_dev(si->fd, si->fn, (void*)&pmsg, 
					   offsetof(PersistentMessage_t, msg));		
		if (0 == err)
			si->status = ERR_BLOCK_END;
		else if (err < 0)
			si->status = err;
		else {
			uint16_t p_crc = pmsg.crc;
			pmsg.crc = 0;
			pmsg.crc = calc_crc16(si->block_magic,
								  (void*)&pmsg, offsetof(PersistentMessage_t, msg));
			if (pmsg.crc == p_crc) {
				/* now get the message */
				err = read_dev(si->fd, si->fn,(void*)&si->msg_queue->msg, pmsg.len);
				if (0 == err && pmsg.len)
					si->status = ERR_BLOCK_END;
				else if (err < 0)
					si->status = err;
				else {
				    p_crc = si->msg_queue->msg.internal.crc;
					si->msg_queue->msg.internal.crc = 0;
					si->msg_queue->msg.internal.crc = 
						calc_crc16(si->block_magic,
								   (void*)&si->msg_queue->msg, pmsg.len);
					if (p_crc != si->msg_queue->msg.internal.crc)
						si->msg_queue->msg.internal.flags |= ERRMEM_FLAG_CRC_ERROR;
					si->status = ERR_OK;
				}
			} else {
				si->status = ERR_BLOCK_END;
				syslog(LOG_CRIT, "%s %s %s", "File ", si->fn, "is corrupted");
			}
		}
		err = seek_dev(si->fd, si->fn, 0, SEEK_CUR);
		if (err < 0)
			si->status = err;
		else
			si->pos = err;
	} else if (si)
		si->status = -EIO;
}

static void get_msg(SessionInfo_t* si)
{
	if (si && si->backend) {
		ErrmemBackendFile_t* dev = (ErrmemBackendFile_t*)si->backend;
		struct stat s = {0};
		/* check existence of file device config file only once per
		 * read request */
		int32_t err = check_ex_file_dev_cfg(dev->name, dev->path);
		if (err < 0) {
			si->state  = ERRMEM_SESSION_BACKEND_ERASED;
			si->status = ERR_SESSION_BACKEND_ERASED;
		} else {
			do {
				err = 0;
				/* Either a valid file is open and ready to for reading
				 * or we have to check the next one */
				if ((si->status == ERR_OK) ||
					(si->status & ERR_BLOCK_END)) {
					if (si->fd >= 0) {
						/* check existence of current file */
						err = stat(si->fn, &s);
						/* If the file no longer exists try to close the file
						 * anyway */
						if (-1 == err) {
							(void)close_dev(si->fd, si->fn);
							si->status = ERR_BLOCK_END;
						} else {
							/* calculate CRC block magic */
							si->block_magic = calc_crc16(0, (void*)si->fn,
									strlen(si->fn));
							/* if it still exists and current pos is not the end,
							 * so try to read */
							if (si->pos < (uint32_t)s.st_size)
								read_message(si);
							else {
								si->pos    = 0;
								si->status = ERR_BLOCK_END;
							}
						}
					}
				}
				/* if status indicates ERR_BLOCK_END get next file and
				 * check it */
				if (si->status & ERR_BLOCK_END) {
					if (-1 != si->fd) {
						err = close_dev(si->fd, si->fn);
						if (err < 0)
							si->status = -err;
						si->fd = -1;
					}
					if (!err) {
						err = get_next_fn(dev, &si->fn, si->seq_num);
						if (err < 0)
							si->status = err;
						else {
							si->block_magic = calc_crc16(0, (void*)si->fn,
															   strlen(si->fn));
							/* In deamon mode dev has the correct values, so
							 * if returned index is greater than high everything
							 * is read.
							 * In client mode check the existence of the file.
							 * Perhaps this new file has been created in the
							 * meanwhile */
							if (!si->client_mode) {
								if ((uint32_t)err > dev->index_high) {
									si->state  = ERRMEM_SESSION_READ_DONE;
									si->status = ERR_READ_END;
								} else {
									si->seq_num = err;
									err = open_dev(si->fn, O_RDONLY, 1);
									if (err >= 0) {
										si->pos    = 0;
										si->fd     = err;
										si->status = ERR_BLOCK_END;
									} else {
										/* The read process is not at the
										 * estimated end. Check for next
										 * file */
										si->pos = 0;
										si->status = ERR_BLOCK_END;
									}
								}
							} else {
								si->seq_num = err;
								err = open_dev(si->fn, O_RDONLY, 1);
								if (err >= 0) {
									si->fd  = err;
									si->pos = 0;
									si->status = ERR_BLOCK_END;
									if (si->seq_num > si->seq_max)
										si->seq_max = si->seq_num;
								} else {
									if (si->seq_num > si->seq_max) {
										si->state  = ERRMEM_SESSION_READ_DONE;
										si->status = ERR_READ_END;
									} else {
										si->pos    = 0;
										si->status = ERR_BLOCK_END;
									}
								}
							}
						}
					}
				} else if (si->status != ERR_OK) {
					if (-1 != si->fd) {
						err = close_dev(si->fd, si->fn);
						if (err < 0)
							si->status = -err;
						si->fd = -1;
					}
				}
			} while (si->status == ERR_BLOCK_END);
		}
	} else
		syslog(LOG_CRIT, "%s %s", "Parameter check failed in get_msg - error =",
			   strerror(-EINVAL));
}

struct filter
{
	regex_t*    regex;
	int32_t     first;
	int32_t     last;
	int32_t     n;
};

// match - match an error memory message against a regular expression
static int32_t match(regex_t* filter, unsigned char* msg)
{
	return !filter || !regexec(filter, (char*) msg, 0, 0, 0);
}


/* dump - Dump all stored messages into a given output backend.
 *        In client mode the session context has to be created.
 *        In server mode the session context is passed to this function.  
 */
static void dump(
	struct ErrmemBackend* backend, struct ErrmemBackend* out,
	struct SessionInfo* s, const char* filter, int32_t first, int32_t last)
{
	SessionInfo_t    si = {0};
	MessageQueue_t** mq = NULL;
	struct filter f = {0};
	regex_t compiled;

	if (out) {
		if (!s) {
			/* This is code not executed by the daemon rather than by application */
			if (backend) 
			{
				if (((ErrmemBackendFile_t*)(backend))->status == STAT_OK) {
					if (((ErrmemBackendFile_t*)(backend))->access_mode == O_RDONLY) {
						si.backend = backend;
						si.fd              = ((ErrmemBackendFile_t*)(si.backend))->fd;
						si.fn              = strdup(((ErrmemBackendFile_t*)(si.backend))->cur_file);
						si.num             = ~0;
						si.num_start       = ~0;
						si.pos             =  0;
						si.seq_num         =  0;
						si.seq_max         = ((ErrmemBackendFile_t*)(si.backend))->index_high;
						si.block_magic     = ((ErrmemBackendFile_t*)(si.backend))->seed;
						si.seq_last        = ~0;
						si.seq_next        = ~0;
						si.seq_new         = ~0;
						si.client_mode     =  1;
						si.persistent      =  0;
						si.msg_queue       = NULL;
						si.status          = ERR_OK;
						si.state           = ERRMEM_SESSION_ESTABLISHED;
						s = &si;
					} else
						syslog(LOG_CRIT, "%s %s %s",
							   "Server / Client mode mixed up. Device: ",
							   backend->get_name(backend), " will not be dumped");				
				}
			} else
				syslog(LOG_CRIT, "%s",
					   "No backend specified for application dump");				
		}

		if (s) {
			f.regex = NULL;
			if (filter && !regcomp(&compiled, filter, REG_NEWLINE))
				f.regex = &compiled;
			if (first > 0)
				f.first = first;
			if (last > 0)
				f.last = last;
			f.n = 0;
		
			do {
				if (!s->msg_queue) {
					s->msg_queue = calloc(1, sizeof(MessageQueue_t));
					if (s->msg_queue) {
						s->backend->read_session(s);
						errmem_create_messages(s);
					}
				}
				mq = errmem_get_message(NULL, &s->msg_queue);
				if (*mq) {
					if (match(f.regex,(*mq)->msg.message) &&
						(!f.last || (f.last && ++f.n <= f.last))) {
						if (out->store)
							out->store(out, offsetof(struct errmem_message, message) + 
									   (*mq)->msg.length, &((*mq)->msg));
					}
					if (mq && *mq) {
						free(*mq);
						*mq = NULL;
						mq  = NULL;
					}
				}
				if (f.last && f.n >= f.last)
					break;
			} while(s->state != ERRMEM_SESSION_READ_DONE &&
					s->state != ERRMEM_SESSION_BACKEND_ERASED &&
					s->status >= 0 && !terminate);
		}
		if (si.fn)
			free(si.fn);
	} else {
		syslog(LOG_CRIT, "%s", "No output backend to dump to specified");
	}
}

static void open_session(SessionInfo_t* si)
{
	int32_t err = 0;
	if (si && si->backend) {
		ErrmemBackendFile_t* dev = (ErrmemBackendFile_t*)si->backend;
		/* get the oldest file if storage exists */
		err = check_ex_file_dev_cfg(dev->name, dev->path);
		if (err < 0)
			si->status = -ENODEV;
		else {
			err = get_next_fn(dev, &si->fn, ~0);
			if (err < 0)
				si->status = err;
			else {
				si->seq_start = err;
				si->seq_num   = err;
				err = open_dev(si->fn, O_RDONLY, 0);
				if (err < 0)
					si->status = err;
				else {
					si->fd  = err;
					si->pos = 0;
					si->seq_max = dev->index_high;
				}
			}
		}						
	} else if (si)
		si->status = -ENODEV;
	else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", "Parameter check failed in open_session"
			   " - error =", strerror(-err));
	}
}

static void read_session(SessionInfo_t* si)
{
	if (si && si->backend) {
		ErrmemBackendFile_t* b = (ErrmemBackendFile_t*)si->backend;
		if (b->status == STAT_OK){
			if (si->state == ERRMEM_SESSION_ESTABLISHED)
				si->state = ERRMEM_SESSION_READ;
			get_msg(si);
		} else if (b->status == STAT_ERASED) {
			si->state  = ERRMEM_SESSION_BACKEND_ERASED;
			si->status = ERR_SESSION_BACKEND_ERASED;
		} else
			si->status = b->status;
	}	
}

ErrmemBackend_t* create_file_structure(ErrmemBackendDev_t* p_dev)
{
	int32_t err = 0;
	ErrmemBackendFile_t* dev = NULL;

	if(p_dev) {
		dev = get_mem(sizeof(ErrmemBackendFile_t));    
		if (dev) {
	
			/* setup the error memory backend interface */
			dev->backend.get_command   = get_command;
			dev->backend.get_name      = get_name;
			dev->backend.get_nr        = get_nr;
			dev->backend.set_nr        = set_nr;
			dev->backend.is_default    = is_default;
			dev->backend.set_default   = set_default;
			dev->backend.get_type      = get_type;
			dev->backend.info          = info;
			dev->backend.store         = store;
			dev->backend.destroy       = destroy;
			dev->backend.create        = create;
			dev->backend.erase         = erase;
			dev->backend.dump          = dump;
			dev->backend.open_session  = open_session;
			dev->backend.read_session  = read_session;

			dev->type            = p_dev->type;
			dev->nr              = p_dev->nr;
			dev->status          = STAT_OK;
			dev->index_high      = ~0;
			dev->index_low       = ~0;
			dev->cnt_files       = 0;
			dev->access_mode     = 0;
			dev->file_size       = ~0;
			dev->seed            = 0;
			dev->storage         = 0;
			/* There is no wrap around here. A normal magic is enough to mark
			 * the beginning of a message */
			dev->msg_magic = MAGIC_MESSAGE;
			/* split name in directory path and filename */
			err = split_name(p_dev->name, &dev->path, &dev->name);
			if (err >= 0) {
				/* + 1 Byte seperator + 10 Byte for max uint32_t value + '\0' */
				dev->name_idx = (char*)get_mem(strlen(dev->name) + 12);
				if (!dev->name_idx) {
					err = -ENOMEM;
					syslog(LOG_CRIT, "%s %s",
						   "Cannot duplicate indexed file name - error = ",
						   strerror(-err));				
				}
				strncpy(dev->name_idx, dev->name, strlen(dev->name) + 12);
			}
			if (err >= 0) {
				if (p_dev->command) {
					dev->command = strndup(p_dev->command,
										   strlen(p_dev->command));
					if (!dev->command) {
						syslog(LOG_INFO, "%s %s %s", "Cannot store command - error =",
							   strerror(errno), "Command will not be executed!");				
					}
				}
				dev->cfg.size       = p_dev->size;
				dev->cfg.nr_files   = p_dev->cnt;
				dev->cfg.persistent = p_dev->persistent;

				dev->cur_file       = NULL;
				dev->fd             = -1;
				dev->flags          =  0;

				if (dev->nr == 1)
					dev->def = 1;
				else
					dev->def = 0;
			}

			if (err < 0)
				destroy(&dev->backend);
			else
				return &dev->backend;
		} else {
			err = -ENOMEM;
			syslog(LOG_CRIT, "%s %s", "Cannot create structure for file storage"
				   " - error =", strerror(-err));		
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", "Invalid parameter passed to function "
			   "create_file_structure - error =", strerror(-err));			
	}
	
	return NULL;
}
